// pages/api/me/password.ts
import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { requireSession } from "@/lib/auth";
import { z } from "zod";
import * as bcrypt from "bcryptjs";

const Body = z.object({
  currentPassword: z.string().min(6),
  newPassword: z.string().min(6).max(128),
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await requireSession(req, res);
  if (!session) return;

  if (req.method !== "POST") {
    return res.status(405).json({ ok: false, error: "Method not allowed" });
  }

  const parsed = Body.safeParse(req.body);
  if (!parsed.success) {
    return res
      .status(400)
      .json({
        ok: false,
        error: "Invalid input",
        details: parsed.error.flatten(),
      });
  }

  const userId = (session.user as any).id as string;
  const { currentPassword, newPassword } = parsed.data;

  const user = await prisma.user.findUnique({ where: { id: userId } });
  if (!user)
    return res.status(404).json({ ok: false, error: "User not found" });

  const ok = await bcrypt.compare(currentPassword, user.passwordHash);
  if (!ok)
    return res
      .status(401)
      .json({ ok: false, error: "Current password is incorrect" });

  const passwordHash = await bcrypt.hash(newPassword, 10);
  await prisma.user.update({ where: { id: userId }, data: { passwordHash } });

  return res.status(200).json({ ok: true, message: "Password updated" });
}
